<?php

/**
 * Migration برای ایجاد جداول اصلی پایگاه داده
 * Create main database tables migration
 */

class CreateMainTables
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    public function up(): void
    {
        $this->createArticlesTable();
        $this->createAdminUsersTable();
        $this->createBotStatsTable();
        $this->createSettingsTable();
        $this->createLogsTable();
        $this->createNewsSourcesTable();
        $this->createKeywordsTable();
        $this->createTranslationsTable();
        $this->createNotificationsTable();
    }

    public function down(): void
    {
        $tables = [
            'notifications',
            'translations', 
            'keywords',
            'news_sources',
            'logs',
            'settings',
            'bot_stats',
            'admin_users',
            'articles'
        ];

        foreach ($tables as $table) {
            $this->pdo->exec("DROP TABLE IF EXISTS $table");
        }
    }

    private function createArticlesTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS articles (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            hash_id TEXT UNIQUE NOT NULL,
            title TEXT NOT NULL,
            description TEXT,
            content TEXT,
            url TEXT NOT NULL,
            image_url TEXT,
            source TEXT,
            published_at TIMESTAMP,
            category TEXT DEFAULT 'general',
            language TEXT DEFAULT 'en',
            translated_title TEXT,
            translated_description TEXT,
            translated_content TEXT,
            posted_at TIMESTAMP,
            post_id TEXT,
            views_count INTEGER DEFAULT 0,
            likes_count INTEGER DEFAULT 0,
            shares_count INTEGER DEFAULT 0,
            is_featured BOOLEAN DEFAULT 0,
            priority INTEGER DEFAULT 0,
            tags TEXT,
            metadata TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // ایجاد ایندکس‌ها
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_articles_hash_id ON articles(hash_id)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_articles_category ON articles(category)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_articles_published_at ON articles(published_at)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_articles_posted_at ON articles(posted_at)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_articles_source ON articles(source)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_articles_language ON articles(language)");
    }

    private function createAdminUsersTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS admin_users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER UNIQUE NOT NULL,
            username TEXT,
            full_name TEXT,
            email TEXT,
            permissions TEXT DEFAULT 'read',
            is_active BOOLEAN DEFAULT 1,
            last_login TIMESTAMP,
            login_attempts INTEGER DEFAULT 0,
            locked_until TIMESTAMP,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // ایجاد ایندکس‌ها
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_admin_users_user_id ON admin_users(user_id)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_admin_users_username ON admin_users(username)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_admin_users_is_active ON admin_users(is_active)");
    }

    private function createBotStatsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS bot_stats (
            id INTEGER PRIMARY KEY,
            total_posts INTEGER DEFAULT 0,
            posts_today INTEGER DEFAULT 0,
            posts_this_hour INTEGER DEFAULT 0,
            posts_this_week INTEGER DEFAULT 0,
            posts_this_month INTEGER DEFAULT 0,
            last_post_time TIMESTAMP,
            errors_count INTEGER DEFAULT 0,
            last_error TEXT,
            total_articles_fetched INTEGER DEFAULT 0,
            total_translations INTEGER DEFAULT 0,
            uptime_start TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // درج رکورد اولیه
        $this->pdo->exec("INSERT OR IGNORE INTO bot_stats (id) VALUES (1)");
    }

    private function createSettingsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS settings (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            key TEXT UNIQUE NOT NULL,
            value TEXT NOT NULL,
            type TEXT DEFAULT 'string',
            description TEXT,
            is_public BOOLEAN DEFAULT 0,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // درج تنظیمات پیش‌فرض
        $defaultSettings = [
            ['bot_name', 'ربات اخبار پیشرفته', 'string', 'نام ربات', 1],
            ['bot_description', 'ربات دریافت و ترجمه اخبار', 'string', 'توضیحات ربات', 1],
            ['max_posts_per_hour', '10', 'integer', 'حداکثر پست در ساعت', 0],
            ['max_posts_per_day', '50', 'integer', 'حداکثر پست در روز', 0],
            ['min_news_interval', '5', 'integer', 'حداقل فاصله بین اخبار (دقیقه)', 0],
            ['translation_enabled', '1', 'boolean', 'فعال بودن ترجمه', 0],
            ['auto_posting_enabled', '1', 'boolean', 'فعال بودن ارسال خودکار', 0],
            ['notification_enabled', '1', 'boolean', 'فعال بودن اعلان‌ها', 0]
        ];

        $stmt = $this->pdo->prepare("
            INSERT OR IGNORE INTO settings (key, value, type, description, is_public) 
            VALUES (?, ?, ?, ?, ?)
        ");

        foreach ($defaultSettings as $setting) {
            $stmt->execute($setting);
        }
    }

    private function createLogsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS logs (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            level TEXT NOT NULL,
            message TEXT NOT NULL,
            context TEXT,
            user_id INTEGER,
            ip_address TEXT,
            user_agent TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // ایجاد ایندکس‌ها
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_logs_level ON logs(level)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_logs_created_at ON logs(created_at)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_logs_user_id ON logs(user_id)");
    }

    private function createNewsSourcesTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS news_sources (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            url TEXT NOT NULL,
            type TEXT DEFAULT 'rss',
            category TEXT DEFAULT 'general',
            language TEXT DEFAULT 'en',
            is_active BOOLEAN DEFAULT 1,
            priority INTEGER DEFAULT 0,
            last_fetch TIMESTAMP,
            fetch_count INTEGER DEFAULT 0,
            error_count INTEGER DEFAULT 0,
            last_error TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // درج منابع پیش‌فرض
        $defaultSources = [
            ['BBC News', 'https://feeds.bbci.co.uk/news/rss.xml', 'rss', 'general', 'en', 1, 10],
            ['CNN', 'https://rss.cnn.com/rss/edition.rss', 'rss', 'general', 'en', 1, 9],
            ['The Guardian', 'https://www.theguardian.com/world/rss', 'rss', 'general', 'en', 1, 8],
            ['Reuters', 'https://feeds.reuters.com/reuters/topNews', 'rss', 'general', 'en', 1, 7],
            ['تسنیم', 'https://www.tasnimnews.com/fa/rss/feed/0/8/0/%D8%A7%D9%82%D8%AA%D8%B5%D8%A7%D8%AF', 'rss', 'economy', 'fa', 1, 10],
            ['مهر', 'https://www.mehrnews.com/rss', 'rss', 'general', 'fa', 1, 9],
            ['ایسنا', 'https://www.isna.ir/rss', 'rss', 'general', 'fa', 1, 8],
            ['فارس', 'https://www.farsnews.ir/rss', 'rss', 'general', 'fa', 1, 7]
        ];

        $stmt = $this->pdo->prepare("
            INSERT OR IGNORE INTO news_sources (name, url, type, category, language, is_active, priority) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");

        foreach ($defaultSources as $source) {
            $stmt->execute($source);
        }
    }

    private function createKeywordsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS keywords (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            keyword TEXT UNIQUE NOT NULL,
            category TEXT DEFAULT 'general',
            priority INTEGER DEFAULT 0,
            is_active BOOLEAN DEFAULT 1,
            match_count INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // درج کلمات کلیدی پیش‌فرض
        $defaultKeywords = [
            // کریپتو
            ['bitcoin', 'crypto', 10], ['crypto', 'crypto', 9], ['cryptocurrency', 'crypto', 9],
            ['btc', 'crypto', 8], ['ethereum', 'crypto', 8], ['blockchain', 'crypto', 7],
            
            // فارکس
            ['forex', 'forex', 10], ['trading', 'forex', 9], ['fx', 'forex', 8],
            ['currency', 'forex', 7], ['usd', 'forex', 8], ['eur', 'forex', 8],
            
            // اقتصاد
            ['economy', 'economy', 10], ['inflation', 'economy', 9], ['fed', 'economy', 9],
            ['interest rate', 'economy', 8], ['gdp', 'economy', 7], ['stock market', 'economy', 8],
            
            // ایران
            ['iran', 'iran', 10], ['iranian', 'iran', 9], ['persian', 'iran', 8],
            ['tehran', 'iran', 8], ['oil', 'iran', 7], ['sanctions', 'iran', 8],
            
            // سیاست
            ['politics', 'politics', 9], ['election', 'politics', 8], ['government', 'politics', 8],
            ['president', 'politics', 7], ['parliament', 'politics', 7]
        ];

        $stmt = $this->pdo->prepare("
            INSERT OR IGNORE INTO keywords (keyword, category, priority) 
            VALUES (?, ?, ?)
        ");

        foreach ($defaultKeywords as $keyword) {
            $stmt->execute($keyword);
        }
    }

    private function createTranslationsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS translations (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            source_text TEXT NOT NULL,
            translated_text TEXT NOT NULL,
            source_language TEXT NOT NULL,
            target_language TEXT NOT NULL,
            service TEXT DEFAULT 'mymemory',
            confidence REAL DEFAULT 0.0,
            usage_count INTEGER DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(source_text, source_language, target_language)
        )";

        $this->pdo->exec($sql);
        
        // ایجاد ایندکس‌ها
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_translations_source_text ON translations(source_text)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_translations_languages ON translations(source_language, target_language)");
    }

    private function createNotificationsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS notifications (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            type TEXT NOT NULL,
            title TEXT NOT NULL,
            message TEXT NOT NULL,
            data TEXT,
            user_id INTEGER,
            is_read BOOLEAN DEFAULT 0,
            is_sent BOOLEAN DEFAULT 0,
            sent_at TIMESTAMP,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";

        $this->pdo->exec($sql);
        
        // ایجاد ایندکس‌ها
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_notifications_user_id ON notifications(user_id)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_notifications_type ON notifications(type)");
        $this->pdo->exec("CREATE INDEX IF NOT EXISTS idx_notifications_is_read ON notifications(is_read)");
    }
}
