<?php

/**
 * ربات خبری پیشرفته تلگرام - نسخه PHP
 * Advanced Telegram News Bot - PHP Version
 * 
 * نویسنده: AI Assistant
 * تاریخ: 2024
 * نسخه: 2.0.0
 */

// بارگذاری autoloader
require_once __DIR__ . '/vendor/autoload.php';

// بارگذاری کلاس‌های اصلی
require_once __DIR__ . '/src/Config.php';
require_once __DIR__ . '/src/DatabaseManager.php';
require_once __DIR__ . '/src/Models/BaseModel.php';
require_once __DIR__ . '/src/Models/Article.php';
require_once __DIR__ . '/src/Services/NewsFetcher.php';
require_once __DIR__ . '/src/Services/TranslationService.php';
require_once __DIR__ . '/src/Services/TelegramBot.php';

use GuzzleHttp\Client;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

/**
 * کلاس مدیر اصلی ربات
 * Main Bot Manager Class
 */
class NewsBotManager
{
    private DatabaseManager $db;
    private TelegramBot $bot;
    private NewsFetcher $newsFetcher;
    private TranslationService $translator;
    private Logger $logger;
    private bool $running = false;
    private array $stats = [];

    public function __construct()
    {
        // مقداردهی اولیه
        $this->initializeLogger();
        $this->initializeDatabase();
        $this->initializeServices();
        
        $this->logger->info('ربات خبری پیشرفته راه‌اندازی شد');
    }

    /**
     * مقداردهی لاگر
     */
    private function initializeLogger(): void
    {
        $this->logger = new Logger('NewsBotManager');
        $this->logger->pushHandler(new StreamHandler(__DIR__ . '/storage/logs/bot_manager.log', Logger::INFO));
    }

    /**
     * مقداردهی پایگاه داده
     */
    private function initializeDatabase(): void
    {
        $dbPath = Config::get('database.path', __DIR__ . '/storage/database.sqlite');
        $this->db = new DatabaseManager($dbPath);
    }

    /**
     * مقداردهی سرویس‌ها
     */
    private function initializeServices(): void
    {
        $botToken = Config::get('bot.token');
        $channelId = Config::get('bot.channel_id');
        
        if (empty($botToken) || empty($channelId)) {
            throw new Exception('توکن ربات یا شناسه کانال تنظیم نشده است');
        }
        
        $this->bot = new TelegramBot($botToken, $channelId, $this->db);
        $this->newsFetcher = new NewsFetcher($this->db);
        $this->translator = new TranslationService($this->db);
    }

    /**
     * شروع ربات
     */
    public function start(): void
    {
        $this->logger->info('🚀 شروع ربات اخبار...');
        $this->running = true;
        
        // بررسی تنظیمات
        $errors = Config::validate();
        if (!empty($errors)) {
            foreach ($errors as $error) {
                $this->logger->error("خطای تنظیمات: $error");
            }
            throw new Exception('تنظیمات ناقص است');
        }
        
        // تست اتصالات
        $this->testConnections();
        
        // شروع حلقه اصلی
        $this->runMainLoop();
    }

    /**
     * توقف ربات
     */
    public function stop(): void
    {
        $this->logger->info('🛑 توقف ربات...');
        $this->running = false;
    }

    /**
     * حلقه اصلی ربات
     */
    private function runMainLoop(): void
    {
        $this->logger->info('✅ ربات با موفقیت شروع شد');
        
        $lastNewsCheck = 0;
        $lastStatsReset = 0;
        $lastCleanup = 0;
        
        while ($this->running) {
            try {
                $now = time();
                
                // بررسی اخبار جدید
                $newsInterval = Config::get('scheduler.news_check_interval', 30) * 60;
                if ($now - $lastNewsCheck >= $newsInterval) {
                    $this->checkAndPostNews();
                    $lastNewsCheck = $now;
                }
                
                // بازنشانی آمار
                if ($now - $lastStatsReset >= 3600) { // هر ساعت
                    $this->resetStats();
                    $lastStatsReset = $now;
                }
                
                // پاک‌سازی داده‌های قدیمی
                $cleanupInterval = Config::get('scheduler.cleanup_interval', 24) * 3600;
                if ($now - $lastCleanup >= $cleanupInterval) {
                    $this->cleanupOldData();
                    $lastCleanup = $now;
                }
                
                // پردازش پیام‌های خصوصی
                $this->handlePrivateMessages();
                
                // انتظار کوتاه
                sleep(10);
                
            } catch (Exception $e) {
                $this->logger->error('خطا در حلقه اصلی: ' . $e->getMessage());
                sleep(60); // انتظار در صورت خطا
            }
        }
        
        $this->logger->info('✅ ربات متوقف شد');
    }

    /**
     * بررسی و ارسال اخبار جدید
     */
    private function checkAndPostNews(): void
    {
        if (!$this->bot->canPostNow()) {
            $this->logger->info('محدودیت ارسال رسیده - رد کردن بررسی اخبار');
            return;
        }
        
        $this->logger->info('🔍 بررسی اخبار جدید...');
        
        try {
            // دریافت اخبار
            $articles = $this->newsFetcher->fetchAllNews();
            $this->logger->info("📊 مجموع " . count($articles) . " مقاله دریافت شد");
            
            if (empty($articles)) {
                $this->logger->warning('⚠️ هیچ مقاله‌ای دریافت نشد');
                return;
            }
            
            // فیلتر کردن اخبار مهم
            $importantArticles = $this->filterImportantNews($articles);
            $this->logger->info("🎯 " . count($importantArticles) . " مقاله مهم شناسایی شد");
            
            if (empty($importantArticles)) {
                $this->logger->info('ℹ️ هیچ مقاله مهمی برای ارسال یافت نشد');
                return;
            }
            
            // ارسال اخبار
            $postedCount = 0;
            foreach (array_slice($importantArticles, 0, 5) as $article) {
                if (!$this->bot->canPostNow()) {
                    $this->logger->info("⏸️ محدودیت ارسال رسیده - {$postedCount} مقاله ارسال شد");
                    break;
                }
                
                // بررسی تکراری نبودن
                if ($this->isArticleDuplicate($article)) {
                    $this->logger->debug("⏭️ مقاله تکراری رد شد: " . substr($article['title'], 0, 50) . "...");
                    continue;
                }
                
                $this->logger->info("📤 ارسال مقاله: " . substr($article['title'], 0, 50) . "...");
                
                if ($this->bot->sendNewsArticle($article)) {
                    $postedCount++;
                    $this->logger->info("✅ مقاله با موفقیت ارسال شد");
                    
                    // فاصله تصادفی بین پست‌ها
                    $minInterval = Config::get('rate_limits.min_news_interval', 5) * 60;
                    $maxInterval = $minInterval * 2;
                    sleep(rand($minInterval, $maxInterval));
                } else {
                    $this->logger->error("❌ خطا در ارسال مقاله");
                }
            }
            
            $this->logger->info("✅ {$postedCount} خبر جدید ارسال شد");
            
        } catch (Exception $e) {
            $this->logger->error("❌ خطا در بررسی و ارسال اخبار: " . $e->getMessage());
        }
    }

    /**
     * فیلتر کردن اخبار مهم
     */
    private function filterImportantNews(array $articles): array
    {
        $important = [];
        $keywords = Config::get('keywords', []);
        $allKeywords = array_merge(...array_values($keywords));
        
        $this->logger->info("🔍 فیلتر کردن " . count($articles) . " مقاله...");
        
        foreach ($articles as $article) {
            $titleLower = strtolower($article['title']);
            $descLower = strtolower($article['description']);
            
            // بررسی کلمات کلیدی
            $keywordFound = false;
            foreach ($allKeywords as $keyword) {
                if (strpos($titleLower, $keyword) !== false || strpos($descLower, $keyword) !== false) {
                    $keywordFound = true;
                    break;
                }
            }
            
            if ($keywordFound) {
                $important[] = $article;
                $this->logger->debug("✅ مقاله مهم: " . substr($article['title'], 0, 50) . "...");
            } else {
                $this->logger->debug("⏭️ مقاله رد شد: " . substr($article['title'], 0, 50) . "...");
            }
        }
        
        // مرتب‌سازی بر اساس تاریخ انتشار
        usort($important, function($a, $b) {
            return strtotime($b['published_at']) - strtotime($a['published_at']);
        });
        
        $this->logger->info("🎯 " . count($important) . " مقاله مهم از " . count($articles) . " مقاله شناسایی شد");
        
        return array_slice($important, 0, 10);
    }

    /**
     * بررسی تکراری بودن مقاله
     */
    private function isArticleDuplicate(array $article): bool
    {
        try {
            $sql = "SELECT COUNT(*) FROM articles WHERE hash_id = :hash_id AND posted_at IS NOT NULL";
            $stmt = $this->db->query($sql, ['hash_id' => $article['hash_id']]);
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            $this->logger->error("خطا در بررسی تکراری بودن مقاله: " . $e->getMessage());
            return false;
        }
    }

    /**
     * پردازش پیام‌های خصوصی
     */
    private function handlePrivateMessages(): void
    {
        try {
            $updates = $this->bot->getUpdates();
            
            foreach ($updates as $update) {
                if (isset($update['message']) && $update['message']['chat']['type'] === 'private') {
                    $this->bot->handlePrivateMessage($update['message']);
                }
            }
            
        } catch (Exception $e) {
            $this->logger->error("خطا در پردازش پیام‌های خصوصی: " . $e->getMessage());
        }
    }

    /**
     * بازنشانی آمار
     */
    private function resetStats(): void
    {
        try {
            $now = new DateTime();
            
            // بازنشانی آمار ساعتی
            if ($now->format('i') === '00') {
                $this->bot->resetHourlyStats();
                $this->logger->info('آمار ساعتی بازنشانی شد');
            }
            
            // بازنشانی آمار روزانه
            if ($now->format('H:i') === '00:00') {
                $this->bot->resetDailyStats();
                $this->logger->info('آمار روزانه بازنشانی شد');
            }
            
        } catch (Exception $e) {
            $this->logger->error("خطا در بازنشانی آمار: " . $e->getMessage());
        }
    }

    /**
     * پاک‌سازی داده‌های قدیمی
     */
    private function cleanupOldData(): void
    {
        $this->logger->info('🧹 شروع پاک‌سازی داده‌های قدیمی...');
        
        try {
            // پاک‌سازی مقالات قدیمی
            $articleModel = new Article($this->db->getConnection());
            $deletedArticles = $articleModel->cleanup(30);
            
            // پاک‌سازی ترجمه‌های قدیمی
            $deletedTranslations = $this->translator->clearDatabaseCache();
            
            // پاک‌سازی لاگ‌های قدیمی
            $sql = "DELETE FROM logs WHERE created_at < :cutoff_date";
            $cutoffDate = date('Y-m-d H:i:s', strtotime('-30 days'));
            $stmt = $this->db->query($sql, ['cutoff_date' => $cutoffDate]);
            $deletedLogs = $stmt->rowCount();
            
            $totalDeleted = $deletedArticles + $deletedTranslations + $deletedLogs;
            $this->logger->info("✅ پاک‌سازی انجام شد: {$totalDeleted} رکورد حذف شد");
            
        } catch (Exception $e) {
            $this->logger->error("خطا در پاک‌سازی: " . $e->getMessage());
        }
    }

    /**
     * تست اتصالات
     */
    private function testConnections(): void
    {
        $this->logger->info('🧪 تست اتصالات...');
        
        // تست اتصال به تلگرام
        $botInfo = $this->bot->getBotInfo();
        if (!empty($botInfo)) {
            $this->logger->info("✅ اتصال به تلگرام موفق - ربات: " . $botInfo['first_name']);
        } else {
            $this->logger->error("❌ خطا در اتصال به تلگرام");
        }
        
        // تست اتصال به پایگاه داده
        $healthCheck = $this->db->healthCheck();
        if ($healthCheck['status'] === 'healthy') {
            $this->logger->info("✅ اتصال به پایگاه داده موفق");
        } else {
            $this->logger->error("❌ مشکل در پایگاه داده: " . implode(', ', $healthCheck['issues']));
        }
        
        // تست سرویس‌های ترجمه
        $translationTests = $this->translator->testServices();
        foreach ($translationTests as $test) {
            if ($test['status'] === 'success') {
                $this->logger->info("✅ سرویس ترجمه {$test['service']} فعال");
            } else {
                $this->logger->warning("⚠️ مشکل در سرویس ترجمه {$test['service']}: " . $test['error']);
            }
        }
        
        // تست منابع خبری
        $sourceTests = $this->newsFetcher->testConnections();
        $successfulSources = 0;
        foreach ($sourceTests as $test) {
            if ($test['status'] === 'success') {
                $successfulSources++;
            }
        }
        
        $this->logger->info("✅ {$successfulSources} منبع خبری از " . count($sourceTests) . " منبع فعال است");
    }

    /**
     * دریافت وضعیت ربات
     */
    public function getStatus(): array
    {
        return [
            'running' => $this->running,
            'can_post' => $this->bot->canPostNow(),
            'database_health' => $this->db->healthCheck(),
            'translation_stats' => $this->translator->getStats(),
            'source_stats' => $this->newsFetcher->getSourceStats()
        ];
    }

    /**
     * اجرای دستی بررسی اخبار
     */
    public function manualNewsCheck(): array
    {
        $this->logger->info('🔄 اجرای دستی بررسی اخبار...');
        
        try {
            $articles = $this->newsFetcher->fetchAllNews();
            $importantArticles = $this->filterImportantNews($articles);
            
            return [
                'total_articles' => count($articles),
                'important_articles' => count($importantArticles),
                'can_post' => $this->bot->canPostNow(),
                'articles' => array_slice($importantArticles, 0, 5)
            ];
            
        } catch (Exception $e) {
            $this->logger->error("خطا در بررسی دستی اخبار: " . $e->getMessage());
            return [
                'error' => $e->getMessage()
            ];
        }
    }
}

/**
 * تابع اصلی اجرا
 */
function main(): void
{
    echo "🤖 ربات پیشرفته مدیریت کانال اخبار تلگرام (نسخه PHP)\n";
    echo "=" . str_repeat("=", 60) . "\n";
    
    try {
        $botManager = new NewsBotManager();
        
        // بررسی آرگومان‌های خط فرمان
        $args = $argv ?? [];
        
        if (in_array('--test', $args)) {
            // حالت تست
            echo "🧪 اجرای تست‌ها...\n";
            $status = $botManager->getStatus();
            print_r($status);
            return;
        }
        
        if (in_array('--manual', $args)) {
            // اجرای دستی
            echo "🔄 اجرای دستی بررسی اخبار...\n";
            $result = $botManager->manualNewsCheck();
            print_r($result);
            return;
        }
        
        if (in_array('--status', $args)) {
            // نمایش وضعیت
            echo "📊 وضعیت ربات:\n";
            $status = $botManager->getStatus();
            print_r($status);
            return;
        }
        
        // اجرای عادی
        $botManager->start();
        
    } catch (Exception $e) {
        echo "❌ خطا: " . $e->getMessage() . "\n";
        exit(1);
    }
}

// اجرای برنامه
if (php_sapi_name() === 'cli') {
    main();
} else {
    // اجرای تحت وب (برای webhook)
    echo "ربات خبری پیشرفته - نسخه PHP\n";
    echo "برای اجرای کامل، از خط فرمان استفاده کنید:\n";
    echo "php index.php\n";
}
