<?php

/**
 * کلاس مدیریت پایگاه داده پیشرفته
 * Advanced Database Manager with Migration Support
 */

use Monolog\Logger;
use Monolog\Handler\StreamHandler;

class DatabaseManager
{
    private PDO $pdo;
    private string $dbPath;
    private array $migrations = [];
    private Logger $logger;

    public function __construct(string $dbPath)
    {
        $this->dbPath = $dbPath;
        $this->logger = new Logger('DatabaseManager');
        $this->logger->pushHandler(new StreamHandler(__DIR__ . '/../storage/logs/database.log'));
        
        $this->initializeDatabase();
        $this->loadMigrations();
        $this->runMigrations();
    }

    /**
     * مقداردهی اولیه پایگاه داده
     */
    private function initializeDatabase(): void
    {
        try {
            // ایجاد پوشه storage اگر وجود ندارد
            $storageDir = dirname($this->dbPath);
            if (!is_dir($storageDir)) {
                mkdir($storageDir, 0755, true);
            }

            $this->pdo = new PDO("sqlite:" . $this->dbPath);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            
            // فعال‌سازی کلیدهای خارجی
            $this->pdo->exec('PRAGMA foreign_keys = ON');
            
            $this->logger->info('پایگاه داده با موفقیت مقداردهی شد');
        } catch (PDOException $e) {
            $this->logger->error('خطا در مقداردهی پایگاه داده: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * بارگذاری migration ها
     */
    private function loadMigrations(): void
    {
        $migrationsDir = __DIR__ . '/../database/migrations';
        if (!is_dir($migrationsDir)) {
            mkdir($migrationsDir, 0755, true);
        }

        $files = glob($migrationsDir . '/*.php');
        foreach ($files as $file) {
            $migrationName = basename($file, '.php');
            $this->migrations[$migrationName] = $file;
        }
        
        ksort($this->migrations);
    }

    /**
     * اجرای migration ها
     */
    private function runMigrations(): void
    {
        $this->createMigrationsTable();
        
        foreach ($this->migrations as $name => $file) {
            if (!$this->isMigrationRun($name)) {
                $this->runMigration($name, $file);
            }
        }
    }

    /**
     * ایجاد جدول migrations
     */
    private function createMigrationsTable(): void
    {
        $sql = "CREATE TABLE IF NOT EXISTS migrations (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            migration VARCHAR(255) UNIQUE NOT NULL,
            batch INTEGER NOT NULL,
            executed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )";
        
        $this->pdo->exec($sql);
    }

    /**
     * بررسی اجرای migration
     */
    private function isMigrationRun(string $migration): bool
    {
        $stmt = $this->pdo->prepare('SELECT COUNT(*) FROM migrations WHERE migration = ?');
        $stmt->execute([$migration]);
        return $stmt->fetchColumn() > 0;
    }

    /**
     * اجرای migration
     */
    private function runMigration(string $name, string $file): void
    {
        try {
            $this->logger->info("اجرای migration: $name");
            
            require_once $file;
            $className = str_replace('_', '', ucwords($name, '_'));
            
            if (class_exists($className)) {
                $migration = new $className($this->pdo);
                $migration->up();
                
                // ثبت migration
                $batch = $this->getNextBatchNumber();
                $stmt = $this->pdo->prepare('INSERT INTO migrations (migration, batch) VALUES (?, ?)');
                $stmt->execute([$name, $batch]);
                
                $this->logger->info("Migration $name با موفقیت اجرا شد");
            }
        } catch (Exception $e) {
            $this->logger->error("خطا در اجرای migration $name: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * دریافت شماره batch بعدی
     */
    private function getNextBatchNumber(): int
    {
        $stmt = $this->pdo->query('SELECT MAX(batch) FROM migrations');
        $maxBatch = $stmt->fetchColumn();
        return ($maxBatch ?: 0) + 1;
    }

    /**
     * دریافت اتصال PDO
     */
    public function getConnection(): PDO
    {
        return $this->pdo;
    }

    /**
     * اجرای کوئری
     */
    public function query(string $sql, array $params = []): PDOStatement
    {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            $this->logger->error("خطا در اجرای کوئری: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * شروع تراکنش
     */
    public function beginTransaction(): bool
    {
        return $this->pdo->beginTransaction();
    }

    /**
     * تایید تراکنش
     */
    public function commit(): bool
    {
        return $this->pdo->commit();
    }

    /**
     * لغو تراکنش
     */
    public function rollback(): bool
    {
        return $this->pdo->rollback();
    }

    /**
     * دریافت آخرین ID وارد شده
     */
    public function lastInsertId(): string
    {
        return $this->pdo->lastInsertId();
    }

    /**
     * پشتیبان‌گیری از پایگاه داده
     */
    public function backup(string $backupPath = null): bool
    {
        try {
            if (!$backupPath) {
                $backupDir = __DIR__ . '/../storage/backups';
                if (!is_dir($backupDir)) {
                    mkdir($backupDir, 0755, true);
                }
                $backupPath = $backupDir . '/backup_' . date('Y-m-d_H-i-s') . '.sqlite';
            }

            if (copy($this->dbPath, $backupPath)) {
                $this->logger->info("پشتیبان‌گیری با موفقیت انجام شد: $backupPath");
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            $this->logger->error("خطا در پشتیبان‌گیری: " . $e->getMessage());
            return false;
        }
    }

    /**
     * بازیابی از پشتیبان
     */
    public function restore(string $backupPath): bool
    {
        try {
            if (!file_exists($backupPath)) {
                throw new Exception("فایل پشتیبان یافت نشد: $backupPath");
            }

            // پشتیبان‌گیری از فایل فعلی
            $currentBackup = $this->dbPath . '.backup.' . time();
            copy($this->dbPath, $currentBackup);

            // بازیابی
            if (copy($backupPath, $this->dbPath)) {
                $this->logger->info("بازیابی با موفقیت انجام شد: $backupPath");
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            $this->logger->error("خطا در بازیابی: " . $e->getMessage());
            return false;
        }
    }

    /**
     * بهینه‌سازی پایگاه داده
     */
    public function optimize(): bool
    {
        try {
            $this->pdo->exec('VACUUM');
            $this->pdo->exec('ANALYZE');
            $this->logger->info('بهینه‌سازی پایگاه داده انجام شد');
            return true;
        } catch (Exception $e) {
            $this->logger->error("خطا در بهینه‌سازی: " . $e->getMessage());
            return false;
        }
    }

    /**
     * دریافت آمار پایگاه داده
     */
    public function getStats(): array
    {
        $stats = [];
        
        // اندازه فایل پایگاه داده
        $stats['file_size'] = file_exists($this->dbPath) ? filesize($this->dbPath) : 0;
        
        // تعداد جداول
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM sqlite_master WHERE type='table'");
        $stats['tables_count'] = $stmt->fetchColumn();
        
        // تعداد مقالات
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM articles");
        $stats['articles_count'] = $stmt->fetchColumn();
        
        // تعداد کاربران ادمین
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM admin_users");
        $stats['admin_users_count'] = $stmt->fetchColumn();
        
        return $stats;
    }

    /**
     * پاک‌سازی داده‌های قدیمی
     */
    public function cleanup(int $daysOld = 30): int
    {
        try {
            $cutoffDate = date('Y-m-d H:i:s', strtotime("-$daysOld days"));
            
            // حذف مقالات قدیمی
            $stmt = $this->pdo->prepare('DELETE FROM articles WHERE created_at < ?');
            $stmt->execute([$cutoffDate]);
            $deletedArticles = $stmt->rowCount();
            
            // حذف لاگ‌های قدیمی
            $stmt = $this->pdo->prepare('DELETE FROM logs WHERE created_at < ?');
            $stmt->execute([$cutoffDate]);
            $deletedLogs = $stmt->rowCount();
            
            $totalDeleted = $deletedArticles + $deletedLogs;
            $this->logger->info("پاک‌سازی انجام شد: $totalDeleted رکورد حذف شد");
            
            return $totalDeleted;
        } catch (Exception $e) {
            $this->logger->error("خطا در پاک‌سازی: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * بررسی سلامت پایگاه داده
     */
    public function healthCheck(): array
    {
        $health = [
            'status' => 'healthy',
            'issues' => [],
            'recommendations' => []
        ];

        try {
            // بررسی اتصال
            $this->pdo->query('SELECT 1');
            
            // بررسی اندازه فایل
            $fileSize = filesize($this->dbPath);
            if ($fileSize > 100 * 1024 * 1024) { // بیش از 100MB
                $health['issues'][] = 'اندازه فایل پایگاه داده زیاد است';
                $health['recommendations'][] = 'پاک‌سازی داده‌های قدیمی را انجام دهید';
            }
            
            // بررسی تعداد migration ها
            $stmt = $this->pdo->query('SELECT COUNT(*) FROM migrations');
            $migrationsCount = $stmt->fetchColumn();
            if ($migrationsCount === 0) {
                $health['issues'][] = 'هیچ migration اجرا نشده است';
                $health['status'] = 'warning';
            }
            
        } catch (Exception $e) {
            $health['status'] = 'error';
            $health['issues'][] = 'خطا در اتصال به پایگاه داده: ' . $e->getMessage();
        }

        return $health;
    }
}
