<?php
/**
 * اسکریپت اجرای ربات از طریق وب - نسخه بهبود یافته
 * Web-based Bot Runner - Enhanced Version
 * 
 * این فایل برای اجرای ربات از طریق مرورگر طراحی شده است
 * و مشکلات اتصال را حل می‌کند
 */

// تنظیمات برای اجرای وب
ini_set('memory_limit', '128M');
ini_set('max_execution_time', 60);
set_time_limit(60);

// بارگذاری autoloader
require_once __DIR__ . '/vendor/autoload.php';

// بارگذاری کلاس‌های اصلی
require_once __DIR__ . '/src/Config.php';
require_once __DIR__ . '/src/DatabaseManager.php';
require_once __DIR__ . '/src/Models/BaseModel.php';
require_once __DIR__ . '/src/Models/Article.php';
require_once __DIR__ . '/src/Services/NewsFetcher.php';
require_once __DIR__ . '/src/Services/TranslationService.php';
require_once __DIR__ . '/src/Services/TelegramBot.php';

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

/**
 * کلاس اجرای ربات از طریق وب - نسخه بهبود یافته
 */
class EnhancedWebBotRunner
{
    private DatabaseManager $db;
    private TelegramBot $bot;
    private NewsFetcher $newsFetcher;
    private TranslationService $translator;
    private Logger $logger;
    private array $results = [];

    public function __construct()
    {
        $this->initializeLogger();
        $this->initializeDatabase();
        $this->initializeServices();
        
        $this->logger->info('ربات از طریق وب راه‌اندازی شد (نسخه بهبود یافته)');
    }

    /**
     * مقداردهی لاگر
     */
    private function initializeLogger(): void
    {
        $this->logger = new Logger('EnhancedWebBotRunner');
        $this->logger->pushHandler(new StreamHandler(__DIR__ . '/storage/logs/enhanced_web_bot.log', Logger::INFO));
    }

    /**
     * مقداردهی پایگاه داده
     */
    private function initializeDatabase(): void
    {
        try {
            $dbPath = Config::get('database.path', __DIR__ . '/storage/database.sqlite');
            
            // اطمینان از وجود پوشه storage
            $storageDir = dirname($dbPath);
            if (!is_dir($storageDir)) {
                mkdir($storageDir, 0755, true);
            }
            
            $this->db = new DatabaseManager($dbPath);
            $this->logger->info('پایگاه داده با موفقیت مقداردهی شد');
        } catch (Exception $e) {
            $this->logger->error('خطا در مقداردهی پایگاه داده: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * مقداردهی سرویس‌ها
     */
    private function initializeServices(): void
    {
        $botToken = Config::get('bot.token');
        $channelId = Config::get('bot.channel_id');
        
        if (empty($botToken) || empty($channelId)) {
            throw new Exception('توکن ربات یا شناسه کانال تنظیم نشده است');
        }
        
        $this->bot = new TelegramBot($botToken, $channelId, $this->db);
        $this->newsFetcher = new NewsFetcher($this->db);
        $this->translator = new TranslationService($this->db);
    }

    /**
     * تست اتصال به تلگرام با تنظیمات بهبود یافته
     */
    public function testTelegramConnection(): array
    {
        $botToken = Config::get('bot.token');
        
        try {
            // تنظیمات cURL بهبود یافته
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => "https://api.cwrtex.ir/bot{$botToken}/getMe",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
                CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_MAXREDIRS => 3,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_HTTPHEADER => [
                    'Accept: application/json',
                    'Content-Type: application/json'
                ]
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                return [
                    'status' => 'error',
                    'error' => "cURL Error: $error",
                    'suggestion' => 'مشکل اتصال اینترنت هاست - با پشتیبانی هاست تماس بگیرید'
                ];
            }
            
            if ($httpCode !== 200) {
                return [
                    'status' => 'error',
                    'error' => "HTTP Error: $httpCode",
                    'suggestion' => 'مشکل در API تلگرام - لطفاً توکن ربات را بررسی کنید'
                ];
            }
            
            $data = json_decode($response, true);
            
            if (!$data || !$data['ok']) {
                return [
                    'status' => 'error',
                    'error' => 'پاسخ نامعتبر از تلگرام',
                    'suggestion' => 'توکن ربات نامعتبر است'
                ];
            }
            
            return [
                'status' => 'success',
                'bot_name' => $data['result']['first_name'],
                'bot_username' => $data['result']['username'],
                'bot_id' => $data['result']['id']
            ];
            
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'error' => $e->getMessage(),
                'suggestion' => 'مشکل در اتصال - تنظیمات هاست را بررسی کنید'
            ];
        }
    }

    /**
     * تست اتصال به NewsAPI
     */
    public function testNewsAPIConnection(): array
    {
        $apiKey = Config::get('apis.news_api.key');
        
        try {
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => "https://newsapi.org/v2/sources?apiKey={$apiKey}",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
                CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            curl_close($ch);
            
            if ($error) {
                return [
                    'status' => 'error',
                    'error' => "cURL Error: $error"
                ];
            }
            
            $data = json_decode($response, true);
            
            if ($data['status'] === 'ok') {
                return [
                    'status' => 'success',
                    'sources_count' => count($data['sources'] ?? [])
                ];
            } else {
                return [
                    'status' => 'error',
                    'error' => $data['message'] ?? 'خطای نامشخص'
                ];
            }
            
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * تست اتصالات کامل
     */
    public function testConnections(): array
    {
        $results = [];
        
        // تست اتصال به تلگرام
        $results['telegram'] = $this->testTelegramConnection();
        
        // تست اتصال به NewsAPI
        $results['newsapi'] = $this->testNewsAPIConnection();
        
        // تست پایگاه داده
        try {
            $healthCheck = $this->db->healthCheck();
            $results['database'] = $healthCheck;
        } catch (Exception $e) {
            $results['database'] = [
                'status' => 'error',
                'error' => $e->getMessage()
            ];
        }
        
        return $results;
    }

    /**
     * اجرای یک چرخه کامل ربات
     */
    public function runCycle(): array
    {
        $this->results = [
            'start_time' => date('Y-m-d H:i:s'),
            'articles_fetched' => 0,
            'articles_posted' => 0,
            'errors' => [],
            'success' => false,
            'memory_usage' => 0,
            'execution_time' => 0
        ];
        
        $startTime = microtime(true);
        
        try {
            // تست اتصالات قبل از اجرا
            $connections = $this->testConnections();
            
            if ($connections['telegram']['status'] !== 'success') {
                $this->results['errors'][] = 'خطا در اتصال به تلگرام: ' . $connections['telegram']['error'];
                $this->results['message'] = 'خطا در اتصال به تلگرام';
                return $this->results;
            }
            
            // بررسی امکان ارسال
            if (!$this->bot->canPostNow()) {
                $this->results['message'] = 'محدودیت ارسال رسیده';
                $this->logger->info('محدودیت ارسال رسیده - رد کردن بررسی اخبار');
                return $this->results;
            }
            
            // دریافت اخبار
            $this->logger->info('🔍 دریافت اخبار جدید...');
            $articles = $this->newsFetcher->fetchAllNews();
            $this->results['articles_fetched'] = count($articles);
            
            if (empty($articles)) {
                $this->results['message'] = 'هیچ مقاله‌ای دریافت نشد';
                $this->logger->warning('⚠️ هیچ مقاله‌ای دریافت نشد');
                return $this->results;
            }
            
            // فیلتر کردن اخبار مهم
            $importantArticles = $this->filterImportantNews($articles);
            $this->logger->info("🎯 " . count($importantArticles) . " مقاله مهم شناسایی شد");
            
            if (empty($importantArticles)) {
                $this->results['message'] = 'هیچ مقاله مهمی یافت نشد';
                $this->logger->info('ℹ️ هیچ مقاله مهمی برای ارسال یافت نشد');
                return $this->results;
            }
            
            // ارسال اخبار (حداکثر 2 مقاله در هر چرخه)
            $postedCount = 0;
            $maxPosts = min(2, count($importantArticles));
            
            for ($i = 0; $i < $maxPosts; $i++) {
                $article = $importantArticles[$i];
                
                if (!$this->bot->canPostNow()) {
                    $this->logger->info("⏸️ محدودیت ارسال رسیده - {$postedCount} مقاله ارسال شد");
                    break;
                }
                
                // بررسی تکراری نبودن
                if ($this->isArticleDuplicate($article)) {
                    $this->logger->debug("⏭️ مقاله تکراری رد شد: " . substr($article['title'], 0, 50) . "...");
                    continue;
                }
                
                $this->logger->info("📤 ارسال مقاله: " . substr($article['title'], 0, 50) . "...");
                
                if ($this->bot->sendNewsArticle($article)) {
                    $postedCount++;
                    $this->logger->info("✅ مقاله با موفقیت ارسال شد");
                    
                    // فاصله کوتاه بین پست‌ها
                    sleep(5);
                } else {
                    $this->logger->error("❌ خطا در ارسال مقاله");
                    $this->results['errors'][] = "خطا در ارسال مقاله: " . substr($article['title'], 0, 30);
                }
            }
            
            $this->results['articles_posted'] = $postedCount;
            $this->results['success'] = true;
            $this->results['message'] = "{$postedCount} خبر جدید ارسال شد";
            
            $this->logger->info("✅ چرخه با موفقیت تکمیل شد - {$postedCount} خبر ارسال شد");
            
        } catch (Exception $e) {
            $errorMsg = "خطا در چرخه اجرا: " . $e->getMessage();
            $this->logger->error($errorMsg);
            $this->results['errors'][] = $errorMsg;
        }
        
        $this->results['end_time'] = date('Y-m-d H:i:s');
        $this->results['execution_time'] = round(microtime(true) - $startTime, 2);
        $this->results['memory_usage'] = round(memory_get_usage(true) / 1024 / 1024, 2);
        
        return $this->results;
    }

    /**
     * فیلتر کردن اخبار مهم
     */
    private function filterImportantNews(array $articles): array
    {
        $important = [];
        $keywords = Config::get('keywords', []);
        $allKeywords = array_merge(...array_values($keywords));
        
        $this->logger->info("🔍 فیلتر کردن " . count($articles) . " مقاله...");
        
        foreach ($articles as $article) {
            $titleLower = strtolower($article['title']);
            $descLower = strtolower($article['description']);
            
            // بررسی کلمات کلیدی
            $keywordFound = false;
            foreach ($allKeywords as $keyword) {
                if (strpos($titleLower, $keyword) !== false || strpos($descLower, $keyword) !== false) {
                    $keywordFound = true;
                    break;
                }
            }
            
            if ($keywordFound) {
                $important[] = $article;
                $this->logger->debug("✅ مقاله مهم: " . substr($article['title'], 0, 50) . "...");
            }
        }
        
        // مرتب‌سازی بر اساس تاریخ انتشار
        usort($important, function($a, $b) {
            return strtotime($b['published_at']) - strtotime($a['published_at']);
        });
        
        return array_slice($important, 0, 3); // حداکثر 3 مقاله مهم
    }

    /**
     * بررسی تکراری بودن مقاله
     */
    private function isArticleDuplicate(array $article): bool
    {
        try {
            $sql = "SELECT COUNT(*) FROM articles WHERE hash_id = :hash_id AND posted_at IS NOT NULL";
            $stmt = $this->db->query($sql, ['hash_id' => $article['hash_id']]);
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            $this->logger->error("خطا در بررسی تکراری بودن مقاله: " . $e->getMessage());
            return false;
        }
    }

    /**
     * دریافت وضعیت ربات
     */
    public function getStatus(): array
    {
        return [
            'can_post' => $this->bot->canPostNow(),
            'database_health' => $this->db->healthCheck(),
            'translation_stats' => $this->translator->getStats(),
            'source_stats' => $this->newsFetcher->getSourceStats(),
            'memory_usage' => round(memory_get_usage(true) / 1024 / 1024, 2),
            'last_run' => $this->getLastRunTime()
        ];
    }

    /**
     * دریافت زمان آخرین اجرا
     */
    private function getLastRunTime(): string
    {
        try {
            $sql = "SELECT MAX(posted_at) as last_post FROM articles WHERE posted_at IS NOT NULL";
            $stmt = $this->db->query($sql);
            $result = $stmt->fetch();
            return $result['last_post'] ?? 'هنوز پستی ارسال نشده';
        } catch (Exception $e) {
            return 'خطا در دریافت زمان';
        }
    }
}

/**
 * تابع اصلی اجرا
 */
function main(): void
{
    // تنظیم header برای JSON
    header('Content-Type: application/json; charset=utf-8');
    
    try {
        $botRunner = new EnhancedWebBotRunner();
        
        // بررسی پارامترهای درخواست
        $action = $_GET['action'] ?? 'run';
        
        switch ($action) {
            case 'run':
                // اجرای یک چرخه
                $results = $botRunner->runCycle();
                echo json_encode($results, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                break;
                
            case 'status':
                // نمایش وضعیت
                $status = $botRunner->getStatus();
                echo json_encode($status, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                break;
                
            case 'test':
                // تست اتصالات
                $connections = $botRunner->testConnections();
                echo json_encode($connections, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                break;
                
            case 'test-telegram':
                // تست فقط تلگرام
                $telegram = $botRunner->testTelegramConnection();
                echo json_encode($telegram, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                break;
                
            case 'test-newsapi':
                // تست فقط NewsAPI
                $newsapi = $botRunner->testNewsAPIConnection();
                echo json_encode($newsapi, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                break;
                
            default:
                // نمایش راهنما
                $help = [
                    'message' => 'ربات خبری پیشرفته - نسخه وب بهبود یافته',
                    'actions' => [
                        'run' => 'اجرای یک چرخه ربات',
                        'status' => 'نمایش وضعیت ربات',
                        'test' => 'تست اتصالات',
                        'test-telegram' => 'تست فقط تلگرام',
                        'test-newsapi' => 'تست فقط NewsAPI'
                    ],
                    'usage' => [
                        '?action=run' => 'اجرای ربات',
                        '?action=status' => 'وضعیت ربات',
                        '?action=test' => 'تست اتصالات',
                        '?action=test-telegram' => 'تست تلگرام',
                        '?action=test-newsapi' => 'تست NewsAPI'
                    ]
                ];
                echo json_encode($help, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        }
        
    } catch (Exception $e) {
        $error = [
            'error' => true,
            'message' => $e->getMessage(),
            'timestamp' => date('Y-m-d H:i:s'),
            'suggestion' => 'لطفاً تنظیمات را بررسی کنید'
        ];
        echo json_encode($error, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }
}

// اجرای برنامه
main();
