<?php
/**
 * اسکریپت اجرای ربات برای هاست اشتراکی
 * Shared Hosting Bot Runner
 * 
 * این فایل برای اجرای ربات در هاست اشتراکی طراحی شده است
 * و محدودیت‌های هاست را در نظر می‌گیرد
 */

// تنظیمات برای هاست اشتراکی
ini_set('memory_limit', '128M');
ini_set('max_execution_time', 60);
set_time_limit(60);

// بارگذاری autoloader
require_once __DIR__ . '/vendor/autoload.php';

// بارگذاری کلاس‌های اصلی
require_once __DIR__ . '/src/Config.php';
require_once __DIR__ . '/src/DatabaseManager.php';
require_once __DIR__ . '/src/Models/BaseModel.php';
require_once __DIR__ . '/src/Models/Article.php';
require_once __DIR__ . '/src/Services/NewsFetcher.php';
require_once __DIR__ . '/src/Services/TranslationService.php';
require_once __DIR__ . '/src/Services/TelegramBot.php';

use GuzzleHttp\Client;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

/**
 * کلاس اجرای ربات برای هاست اشتراکی
 */
class SharedHostingBotRunner
{
    private DatabaseManager $db;
    private TelegramBot $bot;
    private NewsFetcher $newsFetcher;
    private TranslationService $translator;
    private Logger $logger;
    private array $stats = [];

    public function __construct()
    {
        $this->initializeLogger();
        $this->initializeDatabase();
        $this->initializeServices();
        
        $this->logger->info('ربات برای هاست اشتراکی راه‌اندازی شد');
    }

    /**
     * مقداردهی لاگر
     */
    private function initializeLogger(): void
    {
        $this->logger = new Logger('SharedHostingBot');
        $this->logger->pushHandler(new StreamHandler(__DIR__ . '/storage/logs/shared_hosting.log', Logger::INFO));
    }

    /**
     * مقداردهی پایگاه داده
     */
    private function initializeDatabase(): void
    {
        $dbPath = Config::get('database.path', __DIR__ . '/storage/database.sqlite');
        $this->db = new DatabaseManager($dbPath);
    }

    /**
     * مقداردهی سرویس‌ها
     */
    private function initializeServices(): void
    {
        $botToken = Config::get('bot.token');
        $channelId = Config::get('bot.channel_id');
        
        if (empty($botToken) || empty($channelId)) {
            throw new Exception('توکن ربات یا شناسه کانال تنظیم نشده است');
        }
        
        $this->bot = new TelegramBot($botToken, $channelId, $this->db);
        $this->newsFetcher = new NewsFetcher($this->db);
        $this->translator = new TranslationService($this->db);
    }

    /**
     * اجرای یک چرخه کامل ربات
     */
    public function runSingleCycle(): array
    {
        $this->logger->info('🚀 شروع چرخه اجرای ربات...');
        
        $results = [
            'start_time' => date('Y-m-d H:i:s'),
            'articles_fetched' => 0,
            'articles_posted' => 0,
            'errors' => [],
            'success' => false
        ];
        
        try {
            // بررسی امکان ارسال
            if (!$this->bot->canPostNow()) {
                $this->logger->info('محدودیت ارسال رسیده - رد کردن بررسی اخبار');
                $results['message'] = 'محدودیت ارسال رسیده';
                return $results;
            }
            
            // دریافت اخبار
            $this->logger->info('🔍 دریافت اخبار جدید...');
            $articles = $this->newsFetcher->fetchAllNews();
            $results['articles_fetched'] = count($articles);
            
            if (empty($articles)) {
                $this->logger->warning('⚠️ هیچ مقاله‌ای دریافت نشد');
                $results['message'] = 'هیچ مقاله‌ای دریافت نشد';
                return $results;
            }
            
            // فیلتر کردن اخبار مهم
            $importantArticles = $this->filterImportantNews($articles);
            $this->logger->info("🎯 " . count($importantArticles) . " مقاله مهم شناسایی شد");
            
            if (empty($importantArticles)) {
                $this->logger->info('ℹ️ هیچ مقاله مهمی برای ارسال یافت نشد');
                $results['message'] = 'هیچ مقاله مهمی یافت نشد';
                return $results;
            }
            
            // ارسال اخبار (حداکثر 3 مقاله در هر چرخه)
            $postedCount = 0;
            $maxPosts = min(3, count($importantArticles));
            
            for ($i = 0; $i < $maxPosts; $i++) {
                $article = $importantArticles[$i];
                
                if (!$this->bot->canPostNow()) {
                    $this->logger->info("⏸️ محدودیت ارسال رسیده - {$postedCount} مقاله ارسال شد");
                    break;
                }
                
                // بررسی تکراری نبودن
                if ($this->isArticleDuplicate($article)) {
                    $this->logger->debug("⏭️ مقاله تکراری رد شد: " . substr($article['title'], 0, 50) . "...");
                    continue;
                }
                
                $this->logger->info("📤 ارسال مقاله: " . substr($article['title'], 0, 50) . "...");
                
                if ($this->bot->sendNewsArticle($article)) {
                    $postedCount++;
                    $this->logger->info("✅ مقاله با موفقیت ارسال شد");
                    
                    // فاصله کوتاه بین پست‌ها
                    sleep(10);
                } else {
                    $this->logger->error("❌ خطا در ارسال مقاله");
                    $results['errors'][] = "خطا در ارسال مقاله: " . substr($article['title'], 0, 30);
                }
            }
            
            $results['articles_posted'] = $postedCount;
            $results['success'] = true;
            $results['message'] = "{$postedCount} خبر جدید ارسال شد";
            
            $this->logger->info("✅ چرخه با موفقیت تکمیل شد - {$postedCount} خبر ارسال شد");
            
        } catch (Exception $e) {
            $errorMsg = "خطا در چرخه اجرا: " . $e->getMessage();
            $this->logger->error($errorMsg);
            $results['errors'][] = $errorMsg;
        }
        
        $results['end_time'] = date('Y-m-d H:i:s');
        $results['execution_time'] = microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'];
        
        return $results;
    }

    /**
     * فیلتر کردن اخبار مهم
     */
    private function filterImportantNews(array $articles): array
    {
        $important = [];
        $keywords = Config::get('keywords', []);
        $allKeywords = array_merge(...array_values($keywords));
        
        $this->logger->info("🔍 فیلتر کردن " . count($articles) . " مقاله...");
        
        foreach ($articles as $article) {
            $titleLower = strtolower($article['title']);
            $descLower = strtolower($article['description']);
            
            // بررسی کلمات کلیدی
            $keywordFound = false;
            foreach ($allKeywords as $keyword) {
                if (strpos($titleLower, $keyword) !== false || strpos($descLower, $keyword) !== false) {
                    $keywordFound = true;
                    break;
                }
            }
            
            if ($keywordFound) {
                $important[] = $article;
                $this->logger->debug("✅ مقاله مهم: " . substr($article['title'], 0, 50) . "...");
            }
        }
        
        // مرتب‌سازی بر اساس تاریخ انتشار
        usort($important, function($a, $b) {
            return strtotime($b['published_at']) - strtotime($a['published_at']);
        });
        
        return array_slice($important, 0, 5); // حداکثر 5 مقاله مهم
    }

    /**
     * بررسی تکراری بودن مقاله
     */
    private function isArticleDuplicate(array $article): bool
    {
        try {
            $sql = "SELECT COUNT(*) FROM articles WHERE hash_id = :hash_id AND posted_at IS NOT NULL";
            $stmt = $this->db->query($sql, ['hash_id' => $article['hash_id']]);
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            $this->logger->error("خطا در بررسی تکراری بودن مقاله: " . $e->getMessage());
            return false;
        }
    }

    /**
     * دریافت وضعیت ربات
     */
    public function getStatus(): array
    {
        return [
            'can_post' => $this->bot->canPostNow(),
            'database_health' => $this->db->healthCheck(),
            'translation_stats' => $this->translator->getStats(),
            'source_stats' => $this->newsFetcher->getSourceStats(),
            'memory_usage' => memory_get_usage(true),
            'execution_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']
        ];
    }

    /**
     * تست اتصالات
     */
    public function testConnections(): array
    {
        $results = [];
        
        // تست اتصال به تلگرام
        try {
            $botInfo = $this->bot->getBotInfo();
            $results['telegram'] = [
                'status' => 'success',
                'bot_name' => $botInfo['first_name'] ?? 'نامشخص'
            ];
        } catch (Exception $e) {
            $results['telegram'] = [
                'status' => 'error',
                'error' => $e->getMessage()
            ];
        }
        
        // تست اتصال به پایگاه داده
        try {
            $healthCheck = $this->db->healthCheck();
            $results['database'] = $healthCheck;
        } catch (Exception $e) {
            $results['database'] = [
                'status' => 'error',
                'error' => $e->getMessage()
            ];
        }
        
        return $results;
    }
}

/**
 * تابع اصلی اجرا
 */
function main(): void
{
    echo "🤖 ربات خبری پیشرفته - نسخه هاست اشتراکی\n";
    echo "=" . str_repeat("=", 50) . "\n";
    
    try {
        $botRunner = new SharedHostingBotRunner();
        
        // بررسی آرگومان‌های خط فرمان
        $args = $argv ?? [];
        
        if (in_array('--test', $args)) {
            // حالت تست
            echo "🧪 اجرای تست‌ها...\n";
            $status = $botRunner->getStatus();
            $connections = $botRunner->testConnections();
            
            echo "📊 وضعیت ربات:\n";
            print_r($status);
            
            echo "\n🔗 تست اتصالات:\n";
            print_r($connections);
            
            return;
        }
        
        if (in_array('--status', $args)) {
            // نمایش وضعیت
            echo "📊 وضعیت ربات:\n";
            $status = $botRunner->getStatus();
            print_r($status);
            return;
        }
        
        // اجرای یک چرخه
        echo "🚀 اجرای یک چرخه ربات...\n";
        $results = $botRunner->runSingleCycle();
        
        echo "📊 نتایج:\n";
        echo "✅ موفقیت: " . ($results['success'] ? 'بله' : 'خیر') . "\n";
        echo "📰 مقالات دریافت شده: " . $results['articles_fetched'] . "\n";
        echo "📤 مقالات ارسال شده: " . $results['articles_posted'] . "\n";
        echo "⏱️ زمان اجرا: " . round($results['execution_time'], 2) . " ثانیه\n";
        echo "💾 مصرف حافظه: " . round(memory_get_usage(true) / 1024 / 1024, 2) . " MB\n";
        
        if (!empty($results['errors'])) {
            echo "❌ خطاها:\n";
            foreach ($results['errors'] as $error) {
                echo "   - $error\n";
            }
        }
        
        if ($results['success']) {
            echo "\n🎉 چرخه با موفقیت تکمیل شد!\n";
        } else {
            echo "\n⚠️ چرخه با مشکل مواجه شد.\n";
        }
        
    } catch (Exception $e) {
        echo "❌ خطا: " . $e->getMessage() . "\n";
        exit(1);
    }
}

// اجرای برنامه
if (php_sapi_name() === 'cli') {
    main();
} else {
    // اجرای تحت وب (برای تست)
    echo "ربات خبری پیشرفته - نسخه هاست اشتراکی\n";
    echo "برای اجرای کامل، از خط فرمان استفاده کنید:\n";
    echo "php shared_hosting_runner.php\n";
    echo "php shared_hosting_runner.php --test\n";
    echo "php shared_hosting_runner.php --status\n";
}
