<?php

/**
 * کلاس سرویس ترجمه پیشرفته
 * Advanced Translation Service with Caching
 */

use GuzzleHttp\Client;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

class TranslationService
{
    private Client $client;
    private Logger $logger;
    private DatabaseManager $db;
    private array $memoryCache = [];
    private string $service;
    private int $cacheHitCount = 0;
    private int $cacheMissCount = 0;

    public function __construct(DatabaseManager $db)
    {
        $this->db = $db;
        $this->logger = new Logger('TranslationService');
        $this->logger->pushHandler(new StreamHandler(__DIR__ . '/../storage/logs/translation.log'));
        
        $this->client = new Client([
            'timeout' => 15,
            'connect_timeout' => 5,
            'headers' => [
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            ]
        ]);
        
        $this->service = Config::get('apis.translation.service', 'mymemory');
    }

    /**
     * ترجمه متن با کش پیشرفته
     */
    public function translateText(string $text, string $sourceLang = 'en', string $targetLang = 'fa'): string
    {
        if (empty(trim($text)) || strlen(trim($text)) < 3) {
            return $text;
        }

        // بررسی کش حافظه
        $memoryKey = $this->getMemoryCacheKey($text, $sourceLang, $targetLang);
        if (isset($this->memoryCache[$memoryKey])) {
            $this->cacheHitCount++;
            return $this->memoryCache[$memoryKey];
        }

        // بررسی کش پایگاه داده
        $dbTranslation = $this->getFromDatabaseCache($text, $sourceLang, $targetLang);
        if ($dbTranslation) {
            $this->cacheHitCount++;
            $this->memoryCache[$memoryKey] = $dbTranslation;
            return $dbTranslation;
        }

        $this->cacheMissCount++;

        try {
            $translated = $this->performTranslation($text, $sourceLang, $targetLang);
            
            // ذخیره در کش
            $this->saveToDatabaseCache($text, $translated, $sourceLang, $targetLang);
            $this->memoryCache[$memoryKey] = $translated;
            
            return $translated;
            
        } catch (Exception $e) {
            $this->logger->error("خطا در ترجمه: " . $e->getMessage());
            return $text;
        }
    }

    /**
     * ترجمه متن طولانی با تقسیم به بخش‌های کوچک
     */
    public function translateLongText(string $text, string $sourceLang = 'en', string $targetLang = 'fa'): string
    {
        if (strlen($text) <= 500) {
            return $this->translateText($text, $sourceLang, $targetLang);
        }

        // تقسیم متن به جملات
        $sentences = $this->splitIntoSentences($text);
        $translatedSentences = [];

        foreach ($sentences as $sentence) {
            if (strlen(trim($sentence)) > 0) {
                $translatedSentences[] = $this->translateText(trim($sentence), $sourceLang, $targetLang);
            }
        }

        return implode(' ', $translatedSentences);
    }

    /**
     * ترجمه آرایه‌ای از متون
     */
    public function translateBatch(array $texts, string $sourceLang = 'en', string $targetLang = 'fa'): array
    {
        $results = [];
        
        foreach ($texts as $key => $text) {
            $results[$key] = $this->translateText($text, $sourceLang, $targetLang);
            
            // تاخیر کوتاه برای جلوگیری از محدودیت API
            usleep(100000); // 0.1 ثانیه
        }
        
        return $results;
    }

    /**
     * انجام ترجمه با سرویس انتخابی
     */
    private function performTranslation(string $text, string $sourceLang, string $targetLang): string
    {
        switch ($this->service) {
            case 'libretranslate':
                return $this->translateWithLibreTranslate($text, $sourceLang, $targetLang);
            case 'google':
                return $this->translateWithGoogle($text, $sourceLang, $targetLang);
            case 'mymemory':
            default:
                return $this->translateWithMyMemory($text, $sourceLang, $targetLang);
        }
    }

    /**
     * ترجمه با LibreTranslate
     */
    private function translateWithLibreTranslate(string $text, string $sourceLang, string $targetLang): string
    {
        try {
            $url = Config::get('apis.translation.libretranslate_url', 'https://translate.argosopentech.com/translate');
            
            $response = $this->client->post($url, [
                'form_params' => [
                    'q' => $text,
                    'source' => $sourceLang,
                    'target' => $targetLang,
                    'format' => 'text'
                ],
                'timeout' => 15
            ]);
            
            $result = json_decode($response->getBody(), true);
            return $result['translatedText'] ?? $text;
            
        } catch (RequestException $e) {
            $this->logger->warning("خطا در LibreTranslate: " . $e->getMessage());
            // استفاده از سرویس جایگزین
            return $this->translateWithMyMemory($text, $sourceLang, $targetLang);
        }
    }

    /**
     * ترجمه با MyMemory
     */
    private function translateWithMyMemory(string $text, string $sourceLang, string $targetLang): string
    {
        try {
            $response = $this->client->get('https://api.mymemory.translated.net/get', [
                'query' => [
                    'q' => $text,
                    'langpair' => "{$sourceLang}|{$targetLang}"
                ],
                'timeout' => 15
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if ($result['responseStatus'] == 200) {
                $translatedText = $result['responseData']['translatedText'] ?? $text;
                
                // بررسی کیفیت ترجمه
                if ($this->isGoodTranslation($text, $translatedText)) {
                    return $translatedText;
                }
            }
            
            return $text;
            
        } catch (RequestException $e) {
            $this->logger->warning("خطا در MyMemory: " . $e->getMessage());
            return $text;
        }
    }

    /**
     * ترجمه با Google Translate (نیاز به API Key)
     */
    private function translateWithGoogle(string $text, string $sourceLang, string $targetLang): string
    {
        $apiKey = Config::get('apis.translation.google_api_key');
        
        if (empty($apiKey)) {
            $this->logger->warning('کلید Google Translate تنظیم نشده است');
            return $this->translateWithMyMemory($text, $sourceLang, $targetLang);
        }

        try {
            $url = 'https://translation.googleapis.com/language/translate/v2';
            
            $response = $this->client->post($url, [
                'query' => [
                    'key' => $apiKey
                ],
                'form_params' => [
                    'q' => $text,
                    'source' => $sourceLang,
                    'target' => $targetLang,
                    'format' => 'text'
                ],
                'timeout' => 15
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if (isset($result['data']['translations'][0]['translatedText'])) {
                return $result['data']['translations'][0]['translatedText'];
            }
            
            return $text;
            
        } catch (RequestException $e) {
            $this->logger->warning("خطا در Google Translate: " . $e->getMessage());
            return $this->translateWithMyMemory($text, $sourceLang, $targetLang);
        }
    }

    /**
     * دریافت ترجمه از کش پایگاه داده
     */
    private function getFromDatabaseCache(string $text, string $sourceLang, string $targetLang): ?string
    {
        try {
            $sql = "SELECT translated_text, confidence FROM translations 
                    WHERE source_text = :source_text 
                      AND source_language = :source_lang 
                      AND target_language = :target_lang";
            
            $stmt = $this->db->query($sql, [
                'source_text' => $text,
                'source_lang' => $sourceLang,
                'target_lang' => $targetLang
            ]);
            
            $result = $stmt->fetch();
            
            if ($result && $result['confidence'] > 0.5) {
                // به‌روزرسانی تعداد استفاده
                $this->updateUsageCount($text, $sourceLang, $targetLang);
                return $result['translated_text'];
            }
            
            return null;
            
        } catch (Exception $e) {
            $this->logger->error("خطا در دریافت از کش پایگاه داده: " . $e->getMessage());
            return null;
        }
    }

    /**
     * ذخیره ترجمه در کش پایگاه داده
     */
    private function saveToDatabaseCache(string $sourceText, string $translatedText, string $sourceLang, string $targetLang): void
    {
        try {
            $confidence = $this->calculateConfidence($sourceText, $translatedText);
            
            $sql = "INSERT OR REPLACE INTO translations 
                    (source_text, translated_text, source_language, target_language, service, confidence, usage_count) 
                    VALUES (:source_text, :translated_text, :source_lang, :target_lang, :service, :confidence, 1)";
            
            $this->db->query($sql, [
                'source_text' => $sourceText,
                'translated_text' => $translatedText,
                'source_lang' => $sourceLang,
                'target_lang' => $targetLang,
                'service' => $this->service,
                'confidence' => $confidence
            ]);
            
        } catch (Exception $e) {
            $this->logger->error("خطا در ذخیره کش پایگاه داده: " . $e->getMessage());
        }
    }

    /**
     * به‌روزرسانی تعداد استفاده
     */
    private function updateUsageCount(string $text, string $sourceLang, string $targetLang): void
    {
        try {
            $sql = "UPDATE translations 
                    SET usage_count = usage_count + 1, updated_at = CURRENT_TIMESTAMP 
                    WHERE source_text = :source_text 
                      AND source_language = :source_lang 
                      AND target_language = :target_lang";
            
            $this->db->query($sql, [
                'source_text' => $text,
                'source_lang' => $sourceLang,
                'target_lang' => $targetLang
            ]);
            
        } catch (Exception $e) {
            $this->logger->error("خطا در به‌روزرسانی تعداد استفاده: " . $e->getMessage());
        }
    }

    /**
     * محاسبه اعتماد ترجمه
     */
    private function calculateConfidence(string $sourceText, string $translatedText): float
    {
        // بررسی طول متن
        if (strlen($translatedText) < strlen($sourceText) * 0.3) {
            return 0.3;
        }
        
        if (strlen($translatedText) > strlen($sourceText) * 3) {
            return 0.3;
        }
        
        // بررسی وجود کاراکترهای غیرلاتین در ترجمه فارسی
        if (preg_match('/[\x{0600}-\x{06FF}]/u', $translatedText)) {
            return 0.8;
        }
        
        // بررسی شباهت متن اصلی و ترجمه
        if (strtolower($sourceText) === strtolower($translatedText)) {
            return 0.1;
        }
        
        return 0.6;
    }

    /**
     * بررسی کیفیت ترجمه
     */
    private function isGoodTranslation(string $sourceText, string $translatedText): bool
    {
        // بررسی طول متن
        $lengthRatio = strlen($translatedText) / strlen($sourceText);
        if ($lengthRatio < 0.3 || $lengthRatio > 3) {
            return false;
        }
        
        // بررسی شباهت بیش از حد
        if (strtolower($sourceText) === strtolower($translatedText)) {
            return false;
        }
        
        // بررسی وجود کاراکترهای غیرقابل خواندن
        if (preg_match('/[^\x{0000}-\x{007F}\x{0600}-\x{06FF}\s]/u', $translatedText)) {
            return false;
        }
        
        return true;
    }

    /**
     * تقسیم متن به جملات
     */
    private function splitIntoSentences(string $text): array
    {
        // تقسیم بر اساس نقطه، علامت سوال و علامت تعجب
        $sentences = preg_split('/[.!?]+/', $text);
        
        // حذف جملات خالی و کوتاه
        $sentences = array_filter($sentences, function($sentence) {
            return strlen(trim($sentence)) > 10;
        });
        
        return array_map('trim', $sentences);
    }

    /**
     * دریافت کلید کش حافظه
     */
    private function getMemoryCacheKey(string $text, string $sourceLang, string $targetLang): string
    {
        return md5($text . $sourceLang . $targetLang);
    }

    /**
     * پاک‌سازی کش حافظه
     */
    public function clearMemoryCache(): void
    {
        $this->memoryCache = [];
        $this->cacheHitCount = 0;
        $this->cacheMissCount = 0;
    }

    /**
     * پاک‌سازی کش پایگاه داده
     */
    public function clearDatabaseCache(): int
    {
        try {
            $sql = "DELETE FROM translations WHERE created_at < :cutoff_date";
            $cutoffDate = date('Y-m-d H:i:s', strtotime('-30 days'));
            
            $stmt = $this->db->query($sql, ['cutoff_date' => $cutoffDate]);
            return $stmt->rowCount();
            
        } catch (Exception $e) {
            $this->logger->error("خطا در پاک‌سازی کش پایگاه داده: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * دریافت آمار ترجمه
     */
    public function getStats(): array
    {
        $stats = [];
        
        // آمار کش
        $totalRequests = $this->cacheHitCount + $this->cacheMissCount;
        $stats['cache_hit_rate'] = $totalRequests > 0 ? round(($this->cacheHitCount / $totalRequests) * 100, 2) : 0;
        $stats['cache_hits'] = $this->cacheHitCount;
        $stats['cache_misses'] = $this->cacheMissCount;
        
        // آمار پایگاه داده
        try {
            $sql = "SELECT COUNT(*) FROM translations";
            $stmt = $this->db->query($sql);
            $stats['total_translations'] = $stmt->fetchColumn();
            
            $sql = "SELECT COUNT(*) FROM translations WHERE created_at >= DATE('now', '-1 day')";
            $stmt = $this->db->query($sql);
            $stats['translations_today'] = $stmt->fetchColumn();
            
            $sql = "SELECT service, COUNT(*) as count FROM translations GROUP BY service";
            $stmt = $this->db->query($sql);
            $stats['by_service'] = $stmt->fetchAll();
            
        } catch (Exception $e) {
            $this->logger->error("خطا در دریافت آمار ترجمه: " . $e->getMessage());
        }
        
        return $stats;
    }

    /**
     * تست سرویس‌های ترجمه
     */
    public function testServices(): array
    {
        $testText = "Hello, this is a test message for translation.";
        $results = [];
        
        $services = ['mymemory', 'libretranslate'];
        if (Config::get('apis.translation.google_api_key')) {
            $services[] = 'google';
        }
        
        foreach ($services as $service) {
            $originalService = $this->service;
            $this->service = $service;
            
            try {
                $startTime = microtime(true);
                $translated = $this->performTranslation($testText, 'en', 'fa');
                $endTime = microtime(true);
                
                $results[] = [
                    'service' => $service,
                    'status' => 'success',
                    'response_time' => round(($endTime - $startTime) * 1000, 2),
                    'translated_text' => $translated,
                    'quality' => $this->calculateConfidence($testText, $translated)
                ];
                
            } catch (Exception $e) {
                $results[] = [
                    'service' => $service,
                    'status' => 'error',
                    'error' => $e->getMessage()
                ];
            }
            
            $this->service = $originalService;
        }
        
        return $results;
    }
}
